// Content script to track NFPA Publications page visits
// Simple URL pattern matching - no DOM dependencies

function trackNFPAPage() {
    const url = window.location.href;
    
    console.log('NFPA Tracker: Checking URL:', url);
    
    // Extract publication ID from NFPA URL patterns:
    // Standard: https://link.nfpa.org/publications/2/2023
    // Free Access: https://link.nfpa.org/free-access/publications/2/2023
    const standardMatch = url.match(/https:\/\/link\.nfpa\.org\/publications\/([^\/]+)\/([^\/\?]+)/);
    const freeAccessMatch = url.match(/https:\/\/link\.nfpa\.org\/free-access\/publications\/([^\/]+)\/([^\/\?]+)/);
    
    const match = standardMatch || freeAccessMatch;
    
    if (match) {
        const publicationId = match[1];
        const year = match[2];
        const codeBase = `NFPA-${publicationId}-${year}`;
        
        console.log('NFPA Tracker: URL match found - Publication ID:', publicationId, 'Year:', year);
        
        // Check if chrome.storage is available and context is valid
        if (!chrome || !chrome.storage || !chrome.storage.local) {
            console.error('NFPA Tracker: chrome.storage API not available');
            return;
        }
        
        // Additional check for extension context validity
        try {
            if (!chrome.runtime.id) {
                console.log('NFPA Tracker: Extension context invalidated, skipping save');
                return;
            }
        } catch (e) {
            console.log('NFPA Tracker: Extension context invalidated, skipping save');
            return;
        }
        
        // Determine if this is a free-access URL
        const isFreeAccess = url.includes('/free-access/');
        
        // Generate the title directly from URL components
        // Add (FA) suffix for free-access publications
        const pageTitle = isFreeAccess ? `${year} NFPA ${publicationId} (FA)` : `${year} NFPA ${publicationId}`;
        
        const pageData = {
            url: url, // Store the original URL (standard or free-access)
            title: pageTitle,
            publicationId: publicationId,
            year: year,
            codeBase: codeBase, // Same codeBase for both standard and free-access versions
            isFreeAccess: isFreeAccess,
            timestamp: Date.now(),
            visitDate: new Date().toISOString().split('T')[0]
        };
        
        console.log('NFPA Tracker: Saving:', pageData);
        
        // Save immediately
        chrome.storage.local.get(['recentNfpaPublications'], (result) => {
            if (chrome.runtime.lastError) {
                console.error('NFPA Tracker: Error reading storage:', chrome.runtime.lastError);
                
                // Check if it's a context invalidation error
                if (chrome.runtime.lastError.message && 
                    chrome.runtime.lastError.message.includes('context invalidated')) {
                    console.log('NFPA Tracker: Extension context invalidated. Page will be saved on next reload/navigation.');
                    return;
                }
                return;
            }
            
            let recentVisits = result.recentNfpaPublications || [];
            
            // Remove any existing entry for this same page
            recentVisits = recentVisits.filter(item => item.codeBase !== codeBase);
            
            // Add to the beginning of the array
            recentVisits.unshift(pageData);
            
            // Keep only the most recent 50 visits
            recentVisits = recentVisits.slice(0, 50);
            
            // Save back to storage with error handling
            chrome.storage.local.set({ recentNfpaPublications: recentVisits }, () => {
                if (chrome.runtime.lastError) {
                    console.error('NFPA Tracker: Error saving:', chrome.runtime.lastError);
                    
                    // Check if it's a context invalidation error
                    if (chrome.runtime.lastError.message && 
                        chrome.runtime.lastError.message.includes('context invalidated')) {
                        console.log('NFPA Tracker: Extension context invalidated during save. Will retry on next navigation.');
                    }
                } else {
                    console.log('NFPA Tracker: Successfully saved. Total visits:', recentVisits.length);
                }
            });
        });
        
    } else {
        console.log('NFPA Tracker: URL does not match NFPA publications pattern');
    }
}

// Run immediately and watch for URL changes
console.log('NFPA Tracker: Script loaded, running trackNFPAPage...');
trackNFPAPage();

// Watch for URL changes (for single-page applications)
let currentUrl = location.href;
const urlWatcher = setInterval(() => {
    if (location.href !== currentUrl) {
        currentUrl = location.href;
        console.log('NFPA Tracker: URL changed to:', currentUrl);
        trackNFPAPage();
    }
}, 1000);
