/**
 * ICC Family of Solutions Extension - Popup Interface
 * 
 * Main popup window functionality including:
 * - Service navigation menu
 * - Digital Codes submenu with recent/pinned items
 * - Swipe gesture interactions for pin/delete actions
 * - Current service highlighting based on active tab
 */

document.addEventListener('DOMContentLoaded', function() {
    // Get current tab URL to highlight active service
    chrome.tabs.query({active: true, currentWindow: true}, function(tabs) {
        const currentUrl = tabs[0].url;
        highlightCurrentSolution(currentUrl);
    });
    
    // Initialize submenu functionality and load recent codes
    setupSubmenus();
    loadRecentDigitalCodes();
    
    // Set up back button handler for all overlays
    document.addEventListener('click', function(e) {
        if (e.target.classList.contains('back-button')) {
            e.preventDefault();
            e.stopPropagation();
            
            // Close Digital Codes overlay
            const digitalCodesOverlay = document.querySelector('.digitalcodes-overlay');
            if (digitalCodesOverlay && digitalCodesOverlay.classList.contains('show')) {
                digitalCodesOverlay.classList.remove('show');
                return;
            }
            
            // Close cdpACCESS overlay
            const cdpAccessOverlay = document.querySelector('.cdpaccess-overlay');
            if (cdpAccessOverlay && cdpAccessOverlay.classList.contains('show')) {
                cdpAccessOverlay.classList.remove('show');
                return;
            }
            
            // Close ICC Homepage overlay
            const iccSafeOverlay = document.querySelector('.iccsafe-overlay');
            if (iccSafeOverlay && iccSafeOverlay.classList.contains('show')) {
                iccSafeOverlay.classList.remove('show');
                return;
            }
        }
    });
});

/**
 * Toggle pin status for a Digital Code item
 * @param {Element} slot - The slot container element
 * @param {Element} link - The link element containing code data
 */
function togglePin(slot, link) {
    const codeBase = extractCodeBaseFromUrl(link.href);
    
    if (!codeBase) return;
    
    chrome.storage.local.get(['pinnedDigitalCodes'], function(result) {
        let pinnedCodes = result.pinnedDigitalCodes || [];
        const isCurrentlyPinned = pinnedCodes.some(code => code.codeBase === codeBase);
        const pinBtn = slot.querySelector('.pin-btn');
        
        if (isCurrentlyPinned) {
            // Remove from pinned list
            pinnedCodes = pinnedCodes.filter(code => code.codeBase !== codeBase);
            slot.classList.remove('pinned');
            if (pinBtn) pinBtn.classList.remove('pinned');
        } else {
            // Add to pinned list
            const pinnedCode = {
                codeBase: codeBase,
                title: link.textContent,
                url: link.href,
                timestamp: Date.now()
            };
            pinnedCodes.push(pinnedCode);
            slot.classList.add('pinned');
            if (pinBtn) pinBtn.classList.add('pinned');
        }
        
        // Save updated pinned list and refresh display
        chrome.storage.local.set({pinnedDigitalCodes: pinnedCodes}, function() {
            loadRecentDigitalCodes();
        });
    });
}

/**
 * Set up swipe gesture effects for overlay slots
 * Enables left/right hover zones that reveal action buttons
 * @param {Element} container - The overlay container element
 */
function setupSlidingEffects(container) {
    if (!container) return;
    
    const items = container.querySelectorAll('.overlay-slot');
    
    items.forEach((item, index) => {
        const itemContent = item.querySelector('.item-content');
        if (!itemContent) return;
        
        // Clean up existing event listeners
        item.removeEventListener('mousemove', item._mouseMoveHandler);
        item.removeEventListener('mouseleave', item._mouseLeaveHandler);
        
        // Mouse move handler for swipe detection
        item._mouseMoveHandler = (e) => {
            const rect = item.getBoundingClientRect();
            const x = e.clientX - rect.left;
            const width = rect.width;
            const edgeZone = 60; // Pixel width of swipe zones
            
            // Clear previous slide states
            item.classList.remove('slide-left', 'slide-right');
            
            if (x <= edgeZone) {
                // Left edge - show pin button
                item.classList.add('slide-right');
            } else if (x >= width - edgeZone) {
                // Right edge - show delete button
                item.classList.add('slide-left');
            }
        };
        
        // Mouse leave handler to reset state
        item._mouseLeaveHandler = () => {
            item.classList.remove('slide-left', 'slide-right');
            
            // Temporarily disable link clicks during transition
            item._linkClickDisabled = true;
            setTimeout(() => {
                item._linkClickDisabled = false;
            }, 150);
        };
        
        // Attach event listeners
        item.addEventListener('mousemove', item._mouseMoveHandler);
        item.addEventListener('mouseleave', item._mouseLeaveHandler);
        
        // Set up action button click handlers
        const pinBtn = item.querySelector('.pin-btn');
        const deleteBtn = item.querySelector('.delete-btn');
        const link = item.querySelector('.content-layer .nav-link');
        
        if (pinBtn && link) {
            pinBtn.removeEventListener('click', pinBtn._clickHandler);
            pinBtn._clickHandler = (e) => {
                e.preventDefault();
                e.stopPropagation();
                if (!link.classList.contains('disabled')) {
                    togglePin(item, link);
                }
            };
            pinBtn.addEventListener('click', pinBtn._clickHandler);
        }
        
        if (deleteBtn && link) {
            deleteBtn.removeEventListener('click', deleteBtn._clickHandler);
            deleteBtn._clickHandler = (e) => {
                e.preventDefault();
                e.stopPropagation();
                if (!link.classList.contains('disabled')) {
                    deleteItem(link);
                }
            };
            deleteBtn.addEventListener('click', deleteBtn._clickHandler);
        }
    });
}

/**
 * Extract code base identifier from ICC Digital Codes URL
 * @param {string} url - Full URL to extract from
 * @returns {string|null} Code base identifier or null if no match
 */
function extractCodeBaseFromUrl(url) {
    const match = url.match(/https:\/\/codes\.iccsafe\.org\/content\/([^\/]+)/);
    return match ? match[1] : null;
}

/**
 * Delete an item from both recent and pinned storage
 * @param {Element} link - Link element containing the item to delete
 */
function deleteItem(link) {
    const codeBase = extractCodeBaseFromUrl(link.href);
    if (!codeBase) return;
    
    chrome.storage.local.get(['recentDigitalCodes', 'pinnedDigitalCodes'], function(result) {
        let recentPages = result.recentDigitalCodes || [];
        let pinnedPages = result.pinnedDigitalCodes || [];
        
        // Remove from both storage arrays
        recentPages = recentPages.filter(page => page.codeBase !== codeBase);
        pinnedPages = pinnedPages.filter(page => page.codeBase !== codeBase);
        
        chrome.storage.local.set({
            recentDigitalCodes: recentPages,
            pinnedDigitalCodes: pinnedPages
        }, function() {
            loadRecentDigitalCodes();
        });
    });
}

/**
 * Clear all stored recent and pinned items after user confirmation
 */
function clearAllItems() {
    if (confirm('Clear all recent and pinned Digital Codes items?')) {
        chrome.storage.local.remove(['recentDigitalCodes', 'pinnedDigitalCodes'], function() {
            loadRecentDigitalCodes();
        });
    }
}

/**
 * Load and display recent/pinned Digital Codes in the overlay
 * Combines pinned items (shown first) with recent items, up to 5 total
 */
function loadRecentDigitalCodes() {
    chrome.storage.local.get(['recentDigitalCodes', 'pinnedDigitalCodes'], function(result) {
        const recentPages = result.recentDigitalCodes || [];
        const pinnedPages = result.pinnedDigitalCodes || [];
        
        // Get DOM elements
        const overlay = document.querySelector('.digitalcodes-overlay');
        const emptyState = overlay.querySelector('.overlay-empty');
        const availableSlots = overlay.querySelectorAll('.overlay-slot');
        
        // Reset all slots to empty state
        availableSlots.forEach((item, index) => {
            item.style.display = 'none';
            item.classList.remove('show');
            const link = item.querySelector('.content-layer .nav-link');
            
            if (link) {
                link.textContent = '';
                link.href = '#';
                link.removeAttribute('target');
                link.removeAttribute('rel');
                link.classList.add('disabled');
                link.title = '';
            }
            
            item.classList.remove('pinned');
        });
        
        // Combine pinned (first) + recent (excluding already pinned), limit to 5
        const allPages = [...pinnedPages, ...recentPages.filter(page => 
            !pinnedPages.some(pinned => pinned.codeBase === page.codeBase)
        )].slice(0, 5);
        
        // Show/hide empty state based on content
        if (allPages.length === 0) {
            if (emptyState) {
                emptyState.style.display = '';
                emptyState.classList.remove('hidden');
            }
        } else {
            if (emptyState) {
                emptyState.style.display = 'none';
                emptyState.classList.add('hidden');
            }
        }
        
        // Populate slots with combined pages
        allPages.forEach((page, index) => {
            const slot = availableSlots[index];
            
            if (slot) {
                const link = slot.querySelector('.content-layer .nav-link');
                const pinBtn = slot.querySelector('.pin-btn');
                
                if (link) {
                    link.textContent = page.title;
                    link.href = page.url;
                    link.target = '_blank';
                    link.rel = 'noopener';
                    link.classList.remove('disabled');
                    link.title = page.title;
                }
                
                // Mark as pinned if in pinned list
                const isPinned = pinnedPages.some(pinned => pinned.codeBase === page.codeBase);
                if (isPinned) {
                    slot.classList.add('pinned');
                    if (pinBtn) pinBtn.classList.add('pinned');
                } else {
                    slot.classList.remove('pinned');
                    if (pinBtn) pinBtn.classList.remove('pinned');
                }
                
                // Show the populated slot
                slot.style.display = '';
                slot.classList.add('show');
            }
        });
        
        // Re-initialize sliding effects and event handlers
        const overlayContainer = document.querySelector('.digitalcodes-overlay');
        setTimeout(() => setupSlidingEffects(overlayContainer), 100);
        
        setupSubmenus();
        attachClickHandlers();
    });
}

/**
 * Set up submenu functionality and event handlers
 * Uses event delegation to avoid duplicate listeners
 */
function setupSubmenus() {
    // Handle Digital Codes toggle clicks - replace element to avoid duplicates
    document.querySelectorAll('.digitalcodes-toggle').forEach(toggle => {
        const newToggle = toggle.cloneNode(true);
        toggle.parentNode.replaceChild(newToggle, toggle);
        
        newToggle.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            const overlay = document.querySelector('.digitalcodes-overlay');
            if (overlay) {
                overlay.classList.add('show');
            }
        });
    });
    
    // Handle cdpACCESS toggle clicks - replace element to avoid duplicates
    document.querySelectorAll('.cdpaccess-toggle').forEach(toggle => {
        const newToggle = toggle.cloneNode(true);
        toggle.parentNode.replaceChild(newToggle, toggle);
        
        newToggle.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            const overlay = document.querySelector('.cdpaccess-overlay');
            if (overlay) {
                overlay.classList.add('show');
            }
        });
    });
    
    // Handle ICC Homepage toggle clicks - replace element to avoid duplicates
    document.querySelectorAll('.iccsafe-toggle').forEach(toggle => {
        const newToggle = toggle.cloneNode(true);
        toggle.parentNode.replaceChild(newToggle, toggle);
        
        newToggle.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            const overlay = document.querySelector('.iccsafe-overlay');
            if (overlay) {
                overlay.classList.add('show');
            }
        });
    });
    
    // Set up sliding effects for overlay
    const overlayContainer = document.querySelector('.digitalcodes-overlay');
    if (overlayContainer) {
        setupSlidingEffects(overlayContainer);
    }
    
    // Global event handlers using delegation to prevent duplicates
    
    // Pin button handler
    if (!window.pinHandlerAttached) {
        window.pinHandlerAttached = true;
        document.addEventListener('click', function(e) {
            if (e.target.classList.contains('pin-btn') || e.target.closest('.pin-btn')) {
                e.preventDefault();
                e.stopPropagation();
                
                const button = e.target.closest('.pin-btn');
                const slot = button.closest('.overlay-slot');
                const link = slot ? slot.querySelector('.content-layer .nav-link') : null;
                
                if (link && !link.classList.contains('disabled')) {
                    togglePin(slot, link);
                }
            }
        });
    }
    
    // Delete button handler
    if (!window.deleteHandlerAttached) {
        window.deleteHandlerAttached = true;
        document.addEventListener('click', function(e) {
            if (e.target.classList.contains('delete-btn') || e.target.closest('.delete-btn')) {
                e.preventDefault();
                e.stopPropagation();
                
                const button = e.target.closest('.delete-btn');
                const slot = button.closest('.overlay-slot');
                const link = slot ? slot.querySelector('.content-layer .nav-link') : null;
                
                if (link && !link.classList.contains('disabled')) {
                    deleteItem(link);
                }
            }
        });
    }
    
    // Navigation link click prevention during sliding
    if (!window.linkHandlerAttached) {
        window.linkHandlerAttached = true;
        document.addEventListener('click', function(e) {
            if (e.target.classList.contains('nav-link') && e.target.closest('.overlay-slot')) {
                const slot = e.target.closest('.overlay-slot');
                const link = e.target;
                
                // Prevent navigation during swipe actions or if disabled
                if (slot.classList.contains('slide-left') || 
                    slot.classList.contains('slide-right') || 
                    slot._linkClickDisabled ||
                    link.classList.contains('disabled')) {
                    e.preventDefault();
                    return false;
                }
            }
        });
    }
    
    // Clear all button handler
    if (!window.clearHandlerAttached) {
        window.clearHandlerAttached = true;
        document.addEventListener('click', function(e) {
            if (e.target.classList.contains('clear-all-btn')) {
                e.preventDefault();
                e.stopPropagation();
                clearAllItems();
            }
        });
    }
}

/**
 * Highlight the current active service based on the current tab URL
 * @param {string} url - Current tab URL to check against
 */
function highlightCurrentSolution(url) {
    // Remove active class from all menu items
    document.querySelectorAll('.fos-menu-link').forEach(item => {
        item.classList.remove('active');
    });
    
    let activeClass = '';
    
    // Determine which service matches the current URL
    if (url.includes('iccsafe.org') && !url.includes('my.') && !url.includes('codes.') && !url.includes('shop.') && !url.includes('support.')) {
        activeClass = 'fos-iccsafe';
    } else if (url.includes('my.iccsafe.org')) {
        activeClass = 'fos-myicc';
    } else if (url.includes('codes.iccsafe.org')) {
        activeClass = 'fos-digitalcodes';
    } else if (url.includes('cdpaccess.com')) {
        activeClass = 'fos-cdpaccess';
    } else if (url.includes('shop.iccsafe.org')) {
        activeClass = 'fos-store';
    } else if (url.includes('support.iccsafe.org')) {
        activeClass = 'fos-support';
    }
    
    // Apply active styling to matching service
    if (activeClass) {
        const activeElement = document.querySelector(`.${activeClass}`);
        if (activeElement) {
            activeElement.classList.add('active');
        }
    }
}

/**
 * Attach click handlers to close popup after navigation
 * Excludes special buttons that shouldn't close the popup
 */
function attachClickHandlers() {
    // Reset all link event listeners to avoid duplicates
    document.querySelectorAll('.fos-menu-link a:not(.submenu-toggle):not(.digitalcodes-toggle):not(.cdpaccess-toggle):not(.iccsafe-toggle):not(.disabled), .submenu a:not(.back-button):not(.disabled)').forEach(link => {
        link.replaceWith(link.cloneNode(true));
    });
    
    // Attach new listeners to close popup after navigation
    document.querySelectorAll('.fos-menu-link a:not(.submenu-toggle):not(.digitalcodes-toggle):not(.cdpaccess-toggle):not(.iccsafe-toggle):not(.disabled), .submenu a:not(.back-button):not(.disabled)').forEach(link => {
        link.addEventListener('click', function(e) {
            // Skip special buttons that manage the popup itself
            if (this.classList.contains('disabled') || 
                this.classList.contains('submenu-toggle') || 
                this.classList.contains('digitalcodes-toggle') || 
                this.classList.contains('cdpaccess-toggle') || 
                this.classList.contains('iccsafe-toggle') || 
                this.classList.contains('back-button')) {
                return;
            }
            
            // Close popup after brief delay for navigation
            setTimeout(() => {
                window.close();
            }, 100);
        });
    });
}

// Legacy event handler for external links - close popup after navigation
document.querySelectorAll('.fos-menu-link a:not(.submenu-toggle):not(.digitalcodes-toggle):not(.cdpaccess-toggle):not(.iccsafe-toggle):not(.disabled), .submenu a:not(.back-button):not(.disabled)').forEach(link => {
    link.addEventListener('click', function(e) {
        if (this.classList.contains('disabled') || 
            this.classList.contains('submenu-toggle') || 
            this.classList.contains('digitalcodes-toggle') || 
            this.classList.contains('cdpaccess-toggle') || 
            this.classList.contains('iccsafe-toggle') || 
            this.classList.contains('back-button')) {
            return;
        }
        
        setTimeout(() => {
            window.close();
        }, 100);
    });
});
